package aconcorde;

import javax.swing.*;
import javax.swing.table.*;
import javax.swing.event.*;
import javax.swing.border.*;
import java.awt.*;
import java.awt.event.*;
import java.io.*;
import java.text.*;
import java.util.*;
import java.util.List;

public class ConcordanceGUI extends JInternalFrame {

    final static int ALIGNED = 0;
    final static int PLAIN = 1;

    // Global concordance related objects
    private Concordance concordance = new Concordance();
    private String currentWord = "";
    private boolean isCorpusLoaded = false;
    private File currentCorpus = null;
    private String[] encOptions = {"ASCII", "Cp420", "Cp1256", "ISO8859_6", "MacArabic", "UTF8", "UTF-16"};


    private boolean displayAligned = true;

    // Global Swing UI objects 
    private JTextField wordField;
    private JEditorPane editor;
    private JPanel statusBar;
    private JPanel statusBarWords;
    private JPanel freq;
    private JLabel statusFileLabel;
    private JLabel statusWinLabel;
    private JButton doConButton;
    private JSplitPane sp;
    private JTable freqData;
    private JComboBox sortContext;
    final JFileChooser fc = new JFileChooser();
    private JDialog fontDialog;
    private JMenuBar concMenus;
    private TitledBorder concBorder;

    private JSpinner fontSpinner;

    private Font menuFont;
    private Font textFont;
    private Font componentFont;

    // Internationalisation and localisation objects    
    private ResourceBundle labels;
    private Locale currentLocale;
    private ComponentOrientation co;
    private boolean textLeftToRight;

    //private 

    private int wordColIndex = 0;
    private int freqColIndex = 1;

    private Comparator sortContextOrder;

    private ProgressMonitor progressMonitor;

    private StringBuffer outputHTML;

    private static void applyComponentOrientation(Component c, ComponentOrientation o) {

        c.setComponentOrientation(o);

        if (c instanceof JMenu) {
            JMenu menu = (JMenu) c;
            int nComponents = menu.getMenuComponentCount();
            for (int i = 0; i < nComponents; ++i) {
                applyComponentOrientation(menu.getMenuComponent(i), o);
            }
        }
        else if (c instanceof Container) {
            Container container = (Container) c;
            int nComponents = container.getComponentCount();
            for (int i = 0; i < nComponents; ++i) {
                applyComponentOrientation(container.getComponent(i), o);
            }
        }
    }

    public void applyFontSize(Component c, Font f) {

		menuFont = componentFont = textFont = f;

		c.setFont(f);

		if (c instanceof JMenu) {
            JMenu menu = (JMenu) c;
            int nComponents = menu.getMenuComponentCount();
            for (int i = 0; i < nComponents; ++i) {
                applyFontSize(menu.getMenuComponent(i), f);
            }
        }
        else if (c instanceof Container) {
            Container container = (Container) c;
            int nComponents = container.getComponentCount();
            for (int i = 0; i < nComponents; ++i) {
                applyFontSize(container.getComponent(i), f);
            }
        }

    }
   
    private static void applyMenuFontSize(Component c, Font f) {


        if (c instanceof JMenu) {
            c.setFont(f);
            JMenu menu = (JMenu) c;
            int nComponents = menu.getMenuComponentCount();
            for (int i = 0; i < nComponents; ++i) {
                applyMenuFontSize(menu.getMenuComponent(i), f);
            }
        }
        else if (c instanceof JMenuItem) {
            c.setFont(f);
        }

        else if (c instanceof Container) {
            Container container = (Container) c;
            int nComponents = container.getComponentCount();
            for (int i = 0; i < nComponents; ++i) {
                applyMenuFontSize(container.getComponent(i), f);
            }
        }
    }
 
    private static void applyCompFontSize(Component c, Font f) {


        if (c instanceof JMenuItem) {
        }
		else {

			c.setFont(f);

	    	if (c instanceof Container) {
            
       			Container container = (Container) c;
        	    int nComponents = container.getComponentCount();
       		    for (int i = 0; i < nComponents; ++i) {
            	    applyCompFontSize(container.getComponent(i), f);
            	}
        	}
		}
    }

    private void loadResources() {

        try {
            currentLocale = Locale.getDefault();
            labels = ResourceBundle.getBundle("aconcorde.LabelsBundle", currentLocale);
        }
        catch (MissingResourceException mre) {
            mre.printStackTrace(System.out);
            System.exit(1);
        }
    }

    public ConcordanceGUI() {

        setupGUI();
                
    }

    protected Properties programProperties;
    public ConcordanceGUI(File inFile, String enc, int winSize, Font currentFont, Properties programProperties) {

        super(inFile.getName(),
                true,
                true,
                true,
                true);
        
        this.programProperties = programProperties;
    
		menuFont = textFont = componentFont = currentFont;
        initConcordance(inFile, enc);
        concordance.setWindowSize(winSize);
        setupGUI();
        
    }

    public ConcordanceGUI(ConcordanceGUI gui) {

        concordance = gui.getConcordanceClass();
        currentWord = gui.getCurrentWord();
        editor = gui.getEditor();

        
        setupGUI();
        
    }
    

    public Concordance getConcordanceClass() {

        return concordance;
    }

    public String getCurrentWord() {

        return currentWord;
    }

    public JEditorPane getEditor() {

        return editor;
    }

    private void setupGUI() {
        
        // Get the labels for the appropriate locale
        loadResources();

        // Set up the default font
        //menuFont = new Font("Lucida Sans", Font.BOLD, 12);
        //textFont = new Font("Lucida Sans", Font.PLAIN, 12);
        //componentFont = new Font("Lucida Sans", Font.BOLD, 12);

        // Get the current text-direction for the current locale
        co = ComponentOrientation.getOrientation(currentLocale);
        textLeftToRight = co.isLeftToRight();

        sortContextOrder = new RightContextComparator();

        // instantiate layout manager
        BorderLayout layout = new BorderLayout();

        // Create the concordance panel
        JPanel concPanel = new JPanel();

        // get content pane and set the layout
        //setLayout(layout);
        
        concPanel.setLayout(layout);
        // create a panel to hold the word input form

        JPanel inputPanel = new JPanel();
        inputPanel.setLayout(new FlowLayout());
        
        // create textfield object (with label)
        inputPanel.add(new JLabel(labels.getString("labelWord")));
        wordField = new JTextField(20);
        inputPanel.add(wordField);

        // add button. 
        doConButton = new JButton(labels.getString("buttonGetConcordance"));
        doConButton.addActionListener(new DoConcordanceButton());
        inputPanel.add(doConButton);

        // add panel to main container
        concPanel.add(inputPanel, BorderLayout.NORTH);

        // add text area to display concordance

        editor = new JEditorPane();
        editor.setEditable(false);
        editor.setFont(textFont);
        editor.setContentType("text/html");
        
        JPanel editorPanel = new JPanel();
        editorPanel.setLayout(new BorderLayout());
        
        JScrollPane editorScrollPane = new JScrollPane(editor);
        editorScrollPane.setVerticalScrollBarPolicy(JScrollPane.VERTICAL_SCROLLBAR_ALWAYS);

        editorScrollPane.setMinimumSize(new Dimension(10, 10));

        concBorder = BorderFactory.createTitledBorder(labels.getString("tabConcordance"));
        
        editorPanel.setBorder(BorderFactory.createCompoundBorder(
                        concBorder,
                        BorderFactory.createEmptyBorder(5,5,5,5)));
       
        String[] sortContextOptions = {
                labels.getString("sortByLeftAsc"),
                labels.getString("sortByLeftDesc"),
                labels.getString("sortByRightAsc"),
                labels.getString("sortByRightDesc")};
 
        sortContext = new JComboBox(sortContextOptions);
        sortContext.setSelectedIndex(2); // Sorted by left context

        sortContext.addActionListener(new ActionListener() {
			@Override
            public void actionPerformed(ActionEvent e) {

                    
                JComboBox cb = (JComboBox)e.getSource();

                int selectedItem = cb.getSelectedIndex();
                
                
                if (selectedItem == 0) {
                    // User selected order by left context
                    //if (co.isLeftToRight()) {
                    if (textLeftToRight) {
                        sortContextOrder = new LeftContextComparator();
                    }
                    else {
                        sortContextOrder = new RightContextComparator();
                    }
                }
                else if (selectedItem == 1) {
                    // User selected order by left context (descending)
                    //if (co.isLeftToRight()) {
                    if (textLeftToRight) {
                        sortContextOrder = new LeftContextRevComparator();
                    }
                    else {
                        sortContextOrder = new RightContextRevComparator();
                    }

                }
                else if (selectedItem == 2) {
                    // User selected order by right context
                    //if (co.isLeftToRight()) {
                    if (textLeftToRight) {
                        sortContextOrder = new RightContextComparator();
                    }
                    else {
                        sortContextOrder = new LeftContextComparator();
                    }

                }
                else {
                    // order by right context (descending)
                    //if (co.isLeftToRight()) {
                    if (textLeftToRight) {
                        sortContextOrder = new RightContextRevComparator();
                    }
                    else {
                        sortContextOrder = new LeftContextRevComparator();
                    }

                }
               //editor.setText(getConcordance());
                getConcordance();
 
            }
        });

        editorPanel.add(sortContext, BorderLayout.NORTH);
        
        editorPanel.add(editorScrollPane, BorderLayout.CENTER);
        concPanel.add(editorPanel, BorderLayout.CENTER);

        //Create a status bar area
        statusBar = new JPanel();
        statusBar.setLayout(new GridLayout(1,2));
        
        statusFileLabel = new JLabel("");
        statusWinLabel = new JLabel("");
        
        //statusFileLabel.setText(labels.getString("statusbarNoFile") + " ");
        statusWinLabel.setText(labels.getString("statusbarContextSize") + " " + concordance.getWindowSize());
        statusFileLabel.setText(labels.getString("statusbarFile") + " " 
                    + currentCorpus.getName() + " (" + concordance.getEncoding() + ")");
 
        statusFileLabel.setBorder(new BevelBorder(BevelBorder.LOWERED));
        statusWinLabel.setBorder(new BevelBorder(BevelBorder.LOWERED));
        
        statusFileLabel.addMouseListener(new MouseAdapter() {

            public void mouseClicked(MouseEvent event) {
                if (event.getClickCount() >= 2) {
                    createFileOpenDialog();
                }
            }
        });

        // Add a MouseListener to the status WinLabel so that if the
        // user double-clicks, an action equivalent to selecting the
        // Settings/Window Size menu is performed.

        statusWinLabel.addMouseListener(new MouseAdapter() {

			@Override
            public void mouseClicked(MouseEvent event) {
                if (event.getClickCount() >= 2) {
                    createWindowSizeDialog();
                }
            }
        });
        
        statusBar.add(statusFileLabel);
        statusBar.add(statusWinLabel);

        concPanel.add(statusBar,BorderLayout.SOUTH);

		// Initialise splitpane to hold the panels
        
        if (co.isLeftToRight()) {
            sp = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT, freq, concPanel);
        }
        else {
            sp = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT, concPanel, freq);
        }

        // Initialise the Frequency analysis panel

        freq = new JPanel();
        freq = createFrequencyPanel();

		if (co.isLeftToRight()) {
        	sp.setLeftComponent(freq);
        }
        else {
        	sp.setRightComponent(freq);
        }

        
        Container contentPane = getContentPane();

        contentPane.setLayout(new BorderLayout());
        contentPane.add(sp);
        setJMenuBar(createMenuBar());
        applyComponentOrientation(this, co);
		applyFontSize(this, menuFont);
        //applyMenuFontSize(this, menuFont);
        //applyCompFontSize(this, componentFont);

    }

    private JMenuBar createMenuBar() {

        concMenus = new JMenuBar();

        //Create File menu

        JMenu fileMenu = new JMenu(labels.getString("fileMenu"));

        // Create file menu items
        JMenuItem fileSaveItem = new JMenuItem(labels.getString("fileMenuSaveItem"));
        fileSaveItem.addActionListener(new ActionListener() { 
			@Override
            public void actionPerformed(ActionEvent event) {
                createFileSaveDialog();
            }
        });
		
		JMenuItem fileSaveFreqListItem = new JMenuItem(labels.getString("fileMenuSaveFreqListItem"));
        fileSaveFreqListItem.addActionListener(new ActionListener() { 
			@Override
            public void actionPerformed(ActionEvent event) {
                createFileSaveFreqListDialog();
            }
        });

        JMenuItem fileCloseItem = new JMenuItem(labels.getString("fileMenuCloseItem"));

        fileCloseItem.addActionListener(new ActionListener() { 

			@Override
            public void actionPerformed(ActionEvent event) {

                dispose();
                
            }
        });

        JMenuItem filePrintItem = new JMenuItem(labels.getString("fileMenuPrintItem"));
        filePrintItem.setEnabled(false);
        JMenuItem fileCorpusPropItem = new JMenuItem(labels.getString("fileMenuPropItem"));

        fileCorpusPropItem.addActionListener(new ActionListener() {
			@Override
            public void actionPerformed(ActionEvent event) {
                createCorpusPropDialog();
            }
        });

        fileMenu.add(fileSaveItem);
        fileMenu.add(fileSaveFreqListItem);
        fileMenu.add(fileCloseItem);
        fileMenu.add(filePrintItem);
        fileMenu.add(fileCorpusPropItem);

        // Create Edit menu

        JMenu editMenu = new JMenu(labels.getString("editMenu"));

        // Create edit menu items

        JMenuItem editCopyItem = new JMenuItem(labels.getString("editMenuCopyItem"));

        editCopyItem.addActionListener(new ActionListener() { 

			@Override
            public void actionPerformed(ActionEvent event) {

                editor.copy();
                
            }
        });

        JMenuItem editSelectAllItem = new JMenuItem(labels.getString("editMenuSelectAllItem"));

        editSelectAllItem.addActionListener(new ActionListener() { 

            public void actionPerformed(ActionEvent event) {
                editor.requestFocus();
                editor.selectAll();
                
            }
        });

        editMenu.add(editCopyItem);
        editMenu.add(editSelectAllItem);

        // Create View menu

        JMenu viewMenu = new JMenu(labels.getString("viewMenu"));

        // Create view menu items

        // Create display submenu

        JMenu viewDisplayMenu = new JMenu(labels.getString("viewDisplayMenu"));

        ButtonGroup viewDisplayItems = new ButtonGroup();

        JRadioButtonMenuItem viewDisplayAlignedItem = new JRadioButtonMenuItem(labels.getString("viewDisplayAlignedItem"), true);

        viewDisplayAlignedItem.addActionListener(new ActionListener() { 

			@Override
            public void actionPerformed(ActionEvent event) {

                if (!displayAligned) {
                    displayAligned = true;
                    getConcordance();
                    //editor.setText(getConcordance());
                }
                
            }
        });
        
        JRadioButtonMenuItem viewDisplayTextItem = new JRadioButtonMenuItem(labels.getString("viewDisplayTextItem"));
        viewDisplayTextItem.addActionListener(new ActionListener() { 

			@Override
            public void actionPerformed(ActionEvent event) {

                if (displayAligned) {
                    displayAligned = false;
                    getConcordance();
                    //editor.setText(getConcordance());
                }
                
            }
        });

        viewDisplayItems.add(viewDisplayAlignedItem);
        viewDisplayItems.add(viewDisplayTextItem);
        
        viewDisplayMenu.add(viewDisplayAlignedItem);
        viewDisplayMenu.add(viewDisplayTextItem);

        // Create text orientation menu

        JMenu viewOrientationMenu = new JMenu(labels.getString("viewOrientationMenu"));

        ButtonGroup viewOrientationItems = new ButtonGroup();
        
        JRadioButtonMenuItem viewOrientationLTR = new JRadioButtonMenuItem(labels.getString("viewOrientationLTRItem"), co.isLeftToRight());
        
        viewOrientationLTR.addActionListener(new ActionListener() { 

			@Override
            public void actionPerformed(ActionEvent event) {

                if (!textLeftToRight) {
                    textLeftToRight = true;
                    getConcordance();
                    //editor.setText(getConcordance());
                }
                
            }
        });

        
        JRadioButtonMenuItem viewOrientationRTL = new JRadioButtonMenuItem(labels.getString("viewOrientationRTLItem"), !co.isLeftToRight());
        viewOrientationRTL.addActionListener(new ActionListener() { 

			@Override
            public void actionPerformed(ActionEvent event) {

                if (textLeftToRight) {
                    textLeftToRight = false;
                    getConcordance();//editor.setText(getConcordance());
                }
                
            }
        });

        viewOrientationItems.add(viewOrientationLTR);
        viewOrientationItems.add(viewOrientationRTL);
        
        viewOrientationMenu.add(viewOrientationLTR);
        viewOrientationMenu.add(viewOrientationRTL);

        viewMenu.add(viewDisplayMenu);
        viewMenu.add(viewOrientationMenu);

        concMenus.add(fileMenu);
        concMenus.add(editMenu);
        concMenus.add(viewMenu);

        return concMenus;
        
    }

    private JPanel createFrequencyPanel() {

        // The subsequent table component doesn't have text orientation
        // support like many of the other swing components. It works on
        // a left-to-right basis, which is not the intended behaviour if
        // the current locale is right-to-left.
        //
        // Therefore, variables are defined to store the indexes of the
        // 'left' and 'right' column, depending on the current locale.

        // Default values for LTR languages
        
        // Swap for RTL...
        /*if (!co.isLeftToRight()) {
            wordColIndex = 1;
            freqColIndex = 0;
        }*/

        JPanel freqPanel = new JPanel();


        freqData = new JTable(concordance.getWordListSize(), 2) {
                
			@Override
            public boolean isCellEditable(int rowIndex, int vColIndex) {
                return false;

            }

			@Override
            public Component prepareRenderer(TableCellRenderer renderer, 
                    int rowIndex, int vColIndex) {

                Component c = super.prepareRenderer(renderer, rowIndex, vColIndex);
                if (rowIndex % 2 == 0 && !isCellSelected(rowIndex, vColIndex)) {
                    c.setBackground(new Color(239, 249, 255));
                }
                else {
                    // If not shaded, match the table's background
				c.setBackground(getBackground());
				}

				return c;

				}
		};
			
		freqData.getColumnModel().getColumn(wordColIndex).setHeaderValue(labels.getString("columnWord"));
		freqData.getColumnModel().getColumn(freqColIndex).setHeaderValue(labels.getString("columnFreq"));
	
		ListSelectionModel rowSM = freqData.getSelectionModel();
		rowSM.addListSelectionListener(new ListSelectionListener() {

			@Override
			public void valueChanged(ListSelectionEvent e) {

				//Ignore extra messages.
				if (e.getValueIsAdjusting()) return;

				ListSelectionModel lsm = (ListSelectionModel)e.getSource();
				if (lsm.isSelectionEmpty()) {
					//System.out.println("No rows are selected.");
				} else {
					int selectedRow = lsm.getMinSelectionIndex();
					currentWord = (String) freqData.getValueAt(selectedRow, wordColIndex);
					getConcordance();//editor.setText(getConcordance());
				}
			}
		});
		
		freqData.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
		freqData.setAutoCreateColumnsFromModel(false);
		
		String word;
		Integer wordIndex;
		int wordCount = 0;

		ListIterator wordListIt = concordance.getWordListIterator();

		for (int i = 0; i < freqData.getRowCount(); ++i) {

			WordFreq wf = (WordFreq) wordListIt.next();
			freqData.setValueAt(wf.getWord(), i, wordColIndex);
			freqData.setValueAt(new Integer(wf.getWordCount()), i, freqColIndex);
	
		}

		//sortAllRowsBy((DefaultTableModel)freqData.getModel(), 1, false);
		
		freqPanel.setLayout(new BorderLayout());

		// Create a combo box that will allow the user to sort the
		// table by either Word order or frequency order.
		
		String[] sortOptions = {
			labels.getString("sortByWordAsc"),
			labels.getString("sortByWordDesc"),
			labels.getString("sortByFreqAsc"),
			labels.getString("sortByFreqDesc")};


	   
		JComboBox sort = new JComboBox(sortOptions);

		sort.addActionListener(new ActionListener() {

			@Override
			public void actionPerformed(ActionEvent e) {
				
				JComboBox cb = (JComboBox)e.getSource();

				int selectedItem = cb.getSelectedIndex();
				

				if (selectedItem == 0) {
					// User selected order by word
					sortAllRowsBy((DefaultTableModel)freqData.getModel(), wordColIndex, true);
				}
				else if (selectedItem == 1) {
					// User selected order by word (descending)
					sortAllRowsBy((DefaultTableModel)freqData.getModel(), wordColIndex, false);
				}
				else if (selectedItem == 2) {
					// User selected order by frequency
					sortAllRowsBy((DefaultTableModel)freqData.getModel(), freqColIndex, true);
				}
				else {
					// order by frequency (descending)
					sortAllRowsBy((DefaultTableModel)freqData.getModel(), freqColIndex, false);
				}
				
				//sortAllRowsBy((DefaultTableModel)freqData.getModel(), 1, false);
			}
		});

		// Select the first entry in the list and display
		// concordance.
		
		// Select a cell: cell (0,0)
		boolean toggle = false;
		boolean extend = false;
		freqData.setColumnSelectionAllowed(true);
		freqData.setRowSelectionAllowed(true);
		freqData.changeSelection(0, 0, toggle, extend);
		currentWord = (String) freqData.getValueAt(0, 0);
		getConcordance();//editor.setText(getConcordance());


		freqPanel.add(sort, BorderLayout.NORTH); 
		

		freqPanel.add(new JScrollPane(freqData, JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED, 
					JScrollPane.HORIZONTAL_SCROLLBAR_AS_NEEDED), BorderLayout.CENTER);

		// Add a status bar to this panel to show information about
		// the number of unique words, and the total number of words

		//Create a status bar area
		statusBarWords = new JPanel();
		statusBarWords.setLayout(new GridLayout(1,2));
		
		JLabel statusWordsUniqueLabel = new JLabel(labels.getString("statusbarUniqueWords") + " " + freqData.getRowCount());
		JLabel statusWordsTotalLabel = new JLabel(labels.getString("statusbarTotalWords") + " " + concordance.getWordIndexSize());
		
		statusWordsUniqueLabel.setBorder(new BevelBorder(BevelBorder.LOWERED));
		statusWordsTotalLabel.setBorder(new BevelBorder(BevelBorder.LOWERED));
		
		statusBarWords.add(statusWordsUniqueLabel);
		statusBarWords.add(statusWordsTotalLabel);

		freqPanel.add(statusBarWords,BorderLayout.SOUTH);
		
        //applyComponentOrientation(freqPanel, co);
		//applyFontSize(freqPanel, menuFont);

       return freqPanel; 

    }

    public void sortAllRowsBy(DefaultTableModel model, int colIndex, boolean ascending) {
        
        Vector data = model.getDataVector();
        Collections.sort(data, new ColumnSorter(colIndex, ascending));
        model.fireTableStructureChanged();
    }

    private String savePlainConcordanceText() {

        List items = new Vector(concordance.concordanceToList(currentWord));
        
        Iterator i = items.iterator();
        
        String output = "";

        while (i.hasNext()) {
            output += i.next() + "\n";
        }
        
        return output;
    }
	

    private void getHTMLConcordanceText() {

        List items = new Vector(concordance.concordanceToList(currentWord));
		
        Iterator i = items.iterator();
        
        //String outputHTML;

        String concFont = programProperties.getProperty("concordanceFont", "");
        String fontFamily = "sans-serif";
        if (!concFont.equals("")) {
        	fontFamily = ""+concFont+", sans-serif";
        }
        String csscode = "BODY { font-family: "+fontFamily+"; font-size: " + textFont.getSize() + "; }";
        
		csscode += "P {text-align: ";
        if (textLeftToRight) {
           csscode += "left";
        }
        else {
            csscode += "right";
        }
        csscode += "; padding-top: 0; margin-top: 0;}";

        String htmlStart = "<HTML><HEAD><STYLE>" + csscode + "</STYLE></HEAD><BODY><P>";
        String htmlEnd = "</BODY></HTML>";

        outputHTML = new StringBuffer(htmlStart);
		
        while (i.hasNext()) {
            outputHTML.append(i.next() + "<br>\n");
        }
        
        outputHTML.append(htmlEnd);

        editor.setText(outputHTML.toString());
		outputHTML = null;
    }

    private String saveHTMLConcordanceTable() {
            
        List items = new Vector(concordance.concordanceToList(currentWord));

        Collections.sort(items, sortContextOrder);
       
        Iterator i = items.iterator();
        
        String concFont = programProperties.getProperty("concordanceFont", "");
        String fontFamily = "sans-serif";
        if (!concFont.equals("")) {
        	fontFamily = ""+concFont+", sans-serif";
        }
        String csscode = "BODY { font-family: "+fontFamily+"; font-size: " + textFont.getSize() + "}";

        csscode += "P {text-align: ";
        if (textLeftToRight) {
            csscode += "left";
        }
        else {
            csscode += "right";
        }
        csscode += "right; padding-top: 0; margin-top: 0;}";

        csscode += ".leftContext { text-align: left;}";
        csscode += ".rightContext { text-align: right;}";
        csscode += ".word { color: blue;}";

        String htmlStart = "<HTML>\n<HEAD>\n<STYLE>" + csscode + "</STYLE>\n</HEAD>\n<BODY>\n<P>";
        String htmlEnd = "</BODY>\n</HTML>";

        outputHTML = new StringBuffer(htmlStart);

        outputHTML.append("<TABLE align=\"center\">\n");
        while (i.hasNext()) {
            ConcordanceItem c = new ConcordanceItem((ConcordanceItem) i.next());
            outputHTML.append("<TR>\n");
            if (textLeftToRight) {
                outputHTML.append("<TD align=\"right\">").append(c.getLeftContext()).append("</TD>\n");
            }
            else {
                outputHTML.append("<TD align=\"right\">").append(c.getRightContext()).append("</TD>\n");
            }
            outputHTML.append("<TD class=\"word\">").append(c.getWord()).append("</TD>\n");
            if (textLeftToRight) {
                outputHTML.append("<TD align=\"left\">").append(c.getRightContext()).append("</TD>\n");
            }
            else {
                outputHTML.append("<TD align=\"left\">").append(c.getLeftContext()).append("</TD>\n");
            }
            outputHTML.append("</TR>\n");
        }
        
        outputHTML.append("</TABLE>\n");
        
        outputHTML.append(htmlEnd);

        return outputHTML.toString();
    }

    private void getHTMLConcordanceTable() {

        final SwingWorker worker = new SwingWorker() {

			@Override
            public Object construct() {
            
                List items = new Vector(concordance.concordanceToList(currentWord));

                Collections.sort(items, sortContextOrder);
               
                progressMonitor = new ProgressMonitor(ConcordanceGUI.this,
                                          "Getting concordance",
                                          "", 0, items.size());
                progressMonitor.setMillisToDecideToPopup(1000);
                progressMonitor.setProgress(0);
                Iterator i = items.iterator();
                
                String concFont = programProperties.getProperty("concordanceFont", "");
                String fontFamily = "sans-serif";
                if (!concFont.equals("")) {
                	fontFamily = ""+concFont+", sans-serif";
                	
                }
                String csscode = "BODY { font-family: \""+fontFamily+"\";font-size: " + textFont.getSize() + "}";

                csscode += "P {text-align: ";
                if (textLeftToRight) {
                   csscode += "left";
                }
                else {
                    csscode += "right";
                }
                csscode += "right; padding-top: 0; margin-top: 0;}";

                csscode += ".leftContext { text-align: left;}";
                csscode += ".rightContext { text-align: right;}";
                csscode += ".word { color: blue;}";

                String htmlStart = "<HTML>\n<HEAD>\n<STYLE>" + csscode + "</STYLE>\n</HEAD>\n<BODY>\n<P>";
                String htmlEnd = "</BODY>\n</HTML>";

                outputHTML = new StringBuffer(htmlStart);

                outputHTML.append("<TABLE align=\"center\">\n");
                int currentItem = 1;
                while (i.hasNext() && !progressMonitor.isCanceled()) {
                    ConcordanceItem c = new ConcordanceItem((ConcordanceItem) i.next());
                    outputHTML.append("<TR>\n");
                    if (textLeftToRight) {
                        outputHTML.append("<TD align=\"right\">").append(c.getLeftContext()).append("</TD>\n");
                    }
                    else {
                        outputHTML.append("<TD align=\"right\">").append(c.getRightContext()).append("</TD>\n"); 
                    }
                    outputHTML.append("<TD class=\"word\">").append(c.getWord()).append("</TD>\n");
                    if (textLeftToRight) {
                        outputHTML.append("<TD align=\"left\">").append(c.getRightContext()).append("</TD>\n");
                    }
                    else {
                        outputHTML.append("<TD align=\"left\">").append(c.getLeftContext()).append("</TD>\n");
                    }
                    outputHTML.append("</TR>\n");
                    progressMonitor.setProgress(currentItem);
                    currentItem++;
                }
                
                outputHTML.append("</TABLE>\n");

                if (progressMonitor.isCanceled()) {
                    outputHTML.append("<B>Concordance canceled...</B>");
                }
                
                outputHTML.append(htmlEnd);

                progressMonitor.close();
                return ""; // outputHTML; 
            }
            
			@Override
            public void finished() {
                editor.setText(outputHTML.toString());
				//outputHTML = null;
            }
        };
        worker.start();
    }

    private void getConcordance() {

        if (displayAligned) {
            getHTMLConcordanceTable();
        }
        else {
            getHTMLConcordanceText();
        }
                
    }
    
    private class DoConcordanceButton implements ActionListener {
        
		@Override
        public void actionPerformed(ActionEvent event) {

            if (wordField.getText().trim().length() == 0) {
                JOptionPane.showMessageDialog(ConcordanceGUI.this,
                        labels.getString("noWordMessage"),
                        labels.getString("noWordTitle"),
                        JOptionPane.ERROR_MESSAGE);

            }
            else {
				BreakIterator wordIterator = BreakIterator.getWordInstance(currentLocale);
				wordIterator.setText(wordField.getText().trim());
				currentWord = wordField.getText().substring(wordIterator.first(), wordIterator.next());

                getConcordance();//editor.setText(getConcordance());
            }
        }
    }

	private String savePlainWordFrequencyText() {

        StringBuilder output = new StringBuilder();
		for (int i = 0; i < freqData.getModel().getRowCount(); i++) {
			output.append(freqData.getModel().getValueAt(i, 0)).append(", ").append(freqData.getModel().getValueAt(i, 1)).append("\n");
		}      
        return output.toString();
    }
	
	private String saveHtmlWordFrequencyTable(String outEncoding) {

        StringBuilder output = new StringBuilder();
		String csscode = "";
		output.append("<HTML>\n<HEAD><TITLE>"+currentCorpus.getName()+"</TITLE>\n<meta charset='"+outEncoding.toLowerCase()+"'>\n<STYLE>").append(csscode).append("</STYLE>\n</HEAD>\n<BODY>\n<TABLE>");
        
		
		for (int i = 0; i < freqData.getModel().getRowCount(); i++) {
			output.append("<TR>");
			output.append("<TD>").append(freqData.getModel().getValueAt(i, 0)).append("</TD><TD>").append(freqData.getModel().getValueAt(i, 1)).append("</TD>\n");
			output.append("</TR>");
		}
		
		output.append("</TABLE></BODY>\n</HTML>");
        return output.toString();
    }
	
    private void createFileOpenDialog() {


        int returnVal = fc.showOpenDialog(this);

        if (returnVal == JFileChooser.APPROVE_OPTION) {

            // get file name
            File f = fc.getSelectedFile();

            /*String encoding = (String)JOptionPane.showInputDialog(
                    this,
                    labels.getString("encodingDialogMsg"),
                    labels.getString("encodingDialogTitle"),
                    JOptionPane.QUESTION_MESSAGE,
                    null,
                    encOptions,
                    "UTF8");
            */
            
            wordField.setText("");
            isCorpusLoaded = true;
            //initConcordance(f, encoding); 
            //set up frequency data
            freq = new JPanel();
            freq = createFrequencyPanel();
            if (co.isLeftToRight()) {
                sp.setLeftComponent(freq);
            }
            else {
                sp.setRightComponent(freq);
            }

        }

    }

    private void createFileSaveDialog() {

        JFileChooser save = new JFileChooser();
        save.setDialogType(JFileChooser.SAVE_DIALOG);

        GridBagLayout gridbag = new GridBagLayout();
        GridBagConstraints c = new GridBagConstraints();
        c.fill = GridBagConstraints.BOTH;
        c.weightx = 1.0;
        c.ipadx = 2;
        c.ipady = 2;

        JPanel saveOptionsPanel = new JPanel(gridbag);

        saveOptionsPanel.setBorder(new CompoundBorder(
                new CompoundBorder(
                    new EmptyBorder(5, 5, 5, 5),
                    new TitledBorder(labels.getString("saveDialogOptions"))),
                new EmptyBorder(5, 5, 5, 5)));
                    

        JLabel saveTypeLabel = new JLabel(labels.getString("saveDialogSaveType"));
        JLabel saveEncLabel = new JLabel(labels.getString("saveDialogSaveEnc"));
        
        String[] options = {labels.getString("saveDialogSaveAligned"),
                            labels.getString("saveDialogSavePlain") };
        
        JComboBox saveOptions = new JComboBox(options);
        JComboBox encodingOptions = new JComboBox(encOptions);

        saveOptionsPanel.add(saveTypeLabel);
        c.gridwidth = GridBagConstraints.REMAINDER;
        gridbag.setConstraints(saveOptions, c);
        saveOptionsPanel.add(saveOptions);
        c.gridwidth = 1;
        gridbag.setConstraints(saveEncLabel, c);
        saveOptionsPanel.add(saveEncLabel);
        c.gridwidth = GridBagConstraints.REMAINDER;
        gridbag.setConstraints(encodingOptions, c);
        saveOptionsPanel.add(encodingOptions);

        save.setAccessory(saveOptionsPanel);

        if (save.showDialog(this, labels.getString("saveDialogTitle")) != JFileChooser.APPROVE_OPTION) {
            save.remove(saveOptionsPanel);
            return;
        }

        File saveFile = save.getSelectedFile();

        try {
           
            if (saveFile.exists()) {
                if (saveFile.isDirectory()) {
                    // Cannot allow write to happen.
                    // Display an error message
                    
                    JOptionPane.showMessageDialog(this,
                            labels.getString("saveDialogDirectoryError"),
                            labels.getString("saveDialogErrorTitle"),
                            JOptionPane.ERROR_MESSAGE);
                }
                else {
                    // Need some confirmation about overwriting
                    int userChoice = JOptionPane.showConfirmDialog(
                        this,
                        labels.getString("saveDialogOverwriteConfirm"),
                        labels.getString("saveDialogOverwriteTitle"),
                        JOptionPane.YES_NO_OPTION);
                    
                    if (userChoice == JOptionPane.YES_OPTION) {
                        //then overwrite
                        saveConcordance(saveFile, (String) encodingOptions.getSelectedItem(), saveOptions.getSelectedIndex());
                    }
                    else if (userChoice == JOptionPane.NO_OPTION) {
                        //Redisplay save dialog for alternative
                        //selection.
                        //
                        // TODO
                        // Will probably have to implement a
                        // PropertyChangeListener to keep the saveDialog
                        // open until the user successfully saves or
                        // presses Cancel.
                        //
                        // In the meantime, just do nothing if user
                        // selects 'No' to overwriting. They will have
                        // to Save again.
                    }
                }
            }
            else {
                // File doesn't exist, so go ahead and write it...
                saveConcordance(saveFile, (String) encodingOptions.getSelectedItem(), saveOptions.getSelectedIndex());
            }
        

        }
        catch (IOException ioe) {
            System.out.println(ioe);
            
        }

    }
	
	private void createFileSaveFreqListDialog() {

        JFileChooser save = new JFileChooser();
        save.setDialogType(JFileChooser.SAVE_DIALOG);

        GridBagLayout gridbag = new GridBagLayout();
        GridBagConstraints c = new GridBagConstraints();
        c.fill = GridBagConstraints.BOTH;
        c.weightx = 1.0;
        c.ipadx = 2;
        c.ipady = 2;

        JPanel saveOptionsPanel = new JPanel(gridbag);

        saveOptionsPanel.setBorder(new CompoundBorder(
                new CompoundBorder(
                    new EmptyBorder(5, 5, 5, 5),
                    new TitledBorder(labels.getString("saveDialogOptions"))),
                new EmptyBorder(5, 5, 5, 5)));
                    

        JLabel saveTypeLabel = new JLabel(labels.getString("saveDialogSaveType"));
        JLabel saveEncLabel = new JLabel(labels.getString("saveDialogSaveEnc"));
        
        String[] options = {labels.getString("saveDialogSaveAligned"),
                            labels.getString("saveDialogSavePlain") };
        
        JComboBox saveOptions = new JComboBox(options);
        JComboBox encodingOptions = new JComboBox(encOptions);
		encodingOptions.setSelectedItem("UTF8");

        saveOptionsPanel.add(saveTypeLabel);
        c.gridwidth = GridBagConstraints.REMAINDER;
        gridbag.setConstraints(saveOptions, c);
        saveOptionsPanel.add(saveOptions);
        c.gridwidth = 1;
        gridbag.setConstraints(saveEncLabel, c);
        saveOptionsPanel.add(saveEncLabel);
        c.gridwidth = GridBagConstraints.REMAINDER;
        gridbag.setConstraints(encodingOptions, c);
        saveOptionsPanel.add(encodingOptions);

        save.setAccessory(saveOptionsPanel);

        if (save.showDialog(this, labels.getString("saveDialogTitle")) != JFileChooser.APPROVE_OPTION) {
            save.remove(saveOptionsPanel);
            return;
        }

        File saveFile = save.getSelectedFile();
		
		/*switch (saveOptions.getSelectedIndex()) {
			case ConcordanceGUI.ALIGNED:
				if (!(saveFile.getName().toLowerCase().endsWith(".html") || saveFile.getName().toLowerCase().endsWith(".htm"))) {
					saveFile = new File(saveFile.getPath()+".html");
				}
				break;
			case ConcordanceGUI.PLAIN:
				if (!(saveFile.getName().toLowerCase().endsWith(".txt"))) {
					saveFile = new File(saveFile.getPath()+".txt");
				}
				break;
		}*/

        try {
           
            if (saveFile.exists()) {
                if (saveFile.isDirectory()) {
                    // Cannot allow write to happen.
                    // Display an error message
                    
                    JOptionPane.showMessageDialog(this,
                            labels.getString("saveDialogDirectoryError"),
                            labels.getString("saveDialogErrorTitle"),
                            JOptionPane.ERROR_MESSAGE);
                }
                else {
                    // Need some confirmation about overwriting
                    int userChoice = JOptionPane.showConfirmDialog(
                        this,
                        labels.getString("saveDialogOverwriteConfirm"),
                        labels.getString("saveDialogOverwriteTitle"),
                        JOptionPane.YES_NO_OPTION);
                    
                    if (userChoice == JOptionPane.YES_OPTION) {
                        //then overwrite
                        saveFrequencyList(saveFile, (String) encodingOptions.getSelectedItem(), saveOptions.getSelectedIndex());
                    }
                    else if (userChoice == JOptionPane.NO_OPTION) {
                        //Redisplay save dialog for alternative
                        //selection.
                        //
                        // TODO
                        // Will probably have to implement a
                        // PropertyChangeListener to keep the saveDialog
                        // open until the user successfully saves or
                        // presses Cancel.
                        //
                        // In the meantime, just do nothing if user
                        // selects 'No' to overwriting. They will have
                        // to Save again.
                    }
                }
            }
            else {
                // File doesn't exist, so go ahead and write it...
                saveFrequencyList(saveFile, (String) encodingOptions.getSelectedItem(), saveOptions.getSelectedIndex());
            }
        }
        catch (IOException ioe) {
            System.out.println(ioe);   
        }
    }


    private void createCorpusPropDialog() {

        NumberFormat formatter = NumberFormat.getInstance();
        formatter.setMinimumIntegerDigits(1);
        formatter.setMinimumFractionDigits(1);
        formatter.setMaximumFractionDigits(2);

        String message = labels.getString("propertyDialogCorpusFilename") + " " + currentCorpus.getName() + " (" + formatter.format((currentCorpus.length()/1024)) + "KB)" + "\n";

        message += labels.getString("propertyDialogFileEncoding") + " " + concordance.getEncoding() + "\n";
        message += labels.getString("propertyDialogNumUniqueWords") + " " + concordance.getWordListSize() + "\n";
        message += labels.getString("propertyDialogNumWords") + " " + concordance.getWordIndexSize() + "\n";
            

        JOptionPane.showMessageDialog(
                this, 
                message,
                labels.getString("fileMenuPropItem"),
                JOptionPane.PLAIN_MESSAGE);                
    }

    public void saveConcordance(File file, String outEncoding, int type) throws IOException {

        Writer out = 
            new BufferedWriter(new OutputStreamWriter(new FileOutputStream(file), outEncoding));

        if (type == ConcordanceGUI.ALIGNED) {
            out.write(saveHTMLConcordanceTable());
        }
        else if (type == ConcordanceGUI.PLAIN) {
            out.write(savePlainConcordanceText());
        }
        
        out.flush();
        out.close();

    }

	public void saveFrequencyList(File file, String outEncoding, int type) throws IOException {

        Writer out = 
            new BufferedWriter(new OutputStreamWriter(new FileOutputStream(file), outEncoding));

        if (type == ConcordanceGUI.ALIGNED) {
            out.write(saveHtmlWordFrequencyTable(outEncoding));
        }
        else if (type == ConcordanceGUI.PLAIN) {
            out.write(savePlainWordFrequencyText());
        }
        
        out.flush();
        out.close();

    }
		
    private void createWindowSizeDialog() {

        String s = (String) JOptionPane.showInputDialog(
                this,
                labels.getString("windowSizeMessage"),
                labels.getString("windowSizeTitle"),
                JOptionPane.PLAIN_MESSAGE
                );

        if (s == null) {
            //The user selected cancel
            //System.out.println("s is null!");
        }
        else if (s.length() > 0) {
            Integer size = new Integer(s);
            concordance.setWindowSize(size.intValue());
            programProperties.setProperty("defaultWindowSize", size.toString());
            statusWinLabel.setText(labels.getString("statusbarContextSize") + new Integer(concordance.getWindowSize()).toString());

            if (currentWord.length() > 0) {
                getConcordance();//editor.setText(getConcordance());
            }
        }
            

    }

    private void initConcordance(File filename, String enc) {

        concordance = new Concordance();

        try {
            currentCorpus = filename;
            concordance.setEncoding(enc);
            concordance.createWordIndex(filename);
                        //currentWord = "";
            //isCorpusLoaded = true;
            //doConButton.setEnabled(true);
            //sortContext.setEnabled(true);

            //set up frequency data
            /*freq = new JPanel();
            freq = createFrequencyPanel();
            if (co.isLeftToRight()) {
                sp.setLeftComponent(freq);
            }
            else {
                sp.setRightComponent(freq);
            }
*/
            
        }
        catch (IOException e) {
            System.out.println(e);

        }

    }

}
